<?php

namespace App\Http\Requests\Searches;

use App\Rules\ValidateFirmGroup;
use Illuminate\Contracts\Validation\Validator;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Validation\Rule;

class SearchRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $myFirmId = $this->header('firm-id');

        return [
            'search_query' => 'nullable|string',
            'discount' => 'nullable|boolean',
            'sort' => 'nullable|in:price_asc,price_desc,rating_desc',
            'price_min' => 'nullable|numeric|min:0',
            'price_max' => 'nullable|numeric|min:0',
            'page' => 'nullable|integer|min:1',
            'per_page' => 'nullable|integer|min:1|max:100',
            'product_type_id' => 'required|integer|exists:product_name_types,id',
            'supplier_id' => [
                'nullable',
                'integer',
                new ValidateFirmGroup($myFirmId),
            ],
        ];
    }

    public function toParams(): array
    {
        return [
            'firm_id' => $this->header('firm_id'),
            'discount' => $this->boolean('discount'),
            'price_min' => $this->input('price_min'),
            'price_max' => $this->input('price_max'),
            'sort' => $this->input('sort'),
            'page' => $this->integer('page', 1),
            'per_page' => $this->integer('per_page', 10),
            'search_query' => $this->input('search_query'),
            'product_type_id' => $this->input('product_type_id', 1),
            'supplier_id' => $this->input('supplier_id'),
        ];
    }

    public function failedValidation(Validator $validator)
    {
        throw new HttpResponseException(
            response()->json([
                'success' => false,
                'message' => 'Validation Error',
                'Error' => $validator->errors()
            ], 422),
        );
    }
}
