<?php

namespace App\Repositories\Nomenclatures;

use App\Interfaces\Nomenclatures\NomenclatureInterface;
use App\Models\Nomenclature;
use App\Models\NomenclatureIsActive;
use Illuminate\Pagination\LengthAwarePaginator;

class NomenclatureRepository implements NomenclatureInterface
{
    public function get_nomenclatures_by_search($search_query, $product_type_id): LengthAwarePaginator
    {
        $query = Nomenclature::with(['images', 'type', 'manufacturer'])
            ->where('is_active', NomenclatureIsActive::Active);
        if ($product_type_id > 1) {
            $query->where('product_name_type_id', $product_type_id);
        }
        if ($search_query) {
            $query->where(function ($q) use ($search_query) {
                $q->where('product_names.name', 'LIKE', '%' . $search_query . '%')
                    ->orWhere('product_names.mnn', 'LIKE', '%' . $search_query . '%');
            });
        }

        return $query->paginate(10);
    }

    public function get_details_by_id($nomenclature_id)
    {
        return Nomenclature::with(['images' => function ($query) {
            $query->select('product_name_id', 'image_url', 'is_main');
        }])
            ->leftJoin('product_name_types', 'product_name_types.id', '=', 'product_names.product_name_type_id')
            ->leftJoin('manufactures', 'manufactures.id', '=', 'product_names.manufacture_id')
            ->leftJoin('category_product_names', 'category_product_names.id', '=', 'product_names.category_product_id')
            ->select(
                'product_names.id',
                'product_names.name',
                'product_names.mnn',
                'product_names.dose',
                'product_names.amount_text',
                'product_names.description',
                'product_names.full_description',
                'product_names.is_receipt',

                'product_name_types.name as type_name',
                'category_product_names.name as category_name',

                'manufactures.name as manufacture_name',
                'manufactures.country as manufacture_country'
            )
            ->where('product_names.is_active', NomenclatureIsActive::Active)
            ->where('product_names.id', $nomenclature_id)
            ->first();

    }

    public function get_analogues($nomenclature_id): LengthAwarePaginator
    {
        $query= Nomenclature::with(['images', 'manufacturer'])
            ->whereHas('categories', function ($query) use ($nomenclature_id) {
                $query->whereIn('category_product_names.id', function ($subQuery) use ($nomenclature_id) {
                    $subQuery->select('category_product_names_id')
                        ->from('product_name_category_names')
                        ->where('product_names_id', $nomenclature_id);
                });
            })
            ->where('id', '!=', $nomenclature_id)
            ->where('is_active', NomenclatureIsActive::Active)
            ->distinct();

       return $query->paginate(10);
    }

}
