<?php

namespace App\Repositories\OrderRepositories;

use App\Interfaces\OrderInterfaces\OrderInterface;
use App\Models\Firm;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\OrderStatus;
use Illuminate\Support\Facades\DB;

class OrderRepository implements OrderInterface
{
    public function order($firmId, array $partnerId, ?string $search = null, ?string $dateStart = null, ?string $dateEnd = null, ?int $status = null)
    {

        $order = Order::select(
            'orders.id',
            'orders.number',
            'orders.status',
            'firms.title AS firm_title',
            'firms.img AS firm_image',
            'orders.created_at', DB::raw('COUNT(orderlists.id) as items_count')
        )
            ->join('firms', 'orders.id_firm_sender', '=', 'firms.id')
            ->join('orderlists', 'orders.id', '=', 'orderlists.order_id')
            ->where('orders.id_firm_sender', $firmId)
            ->whereIn('orders.id_firm_income', $partnerId)
            ->groupBy('orders.id', 'orders.number', 'orders.status', 'orders.created_at', 'firms.title', 'firms.img')
            ->orderBy('orders.created_at', 'desc');
        if ($search) {
            $order->where('firms.title', 'like', '%' . $search . '%');
        }
        if ($dateStart and $dateEnd) {
            $order->whereBetween('orders.created_at', [$dateStart, $dateEnd]);
        }
        if ($status) {
            $order->where('orders.status', $status);
        } else {
            $order->whereNot('orders.status', OrderStatus::ARCHIVED);
        }

        return $order->paginate(10);
    }


    public function addToOrder($data)
    {
        $order = Order::create($data);
        return $order;
    }

    public function addToOrderItem($data)
    {
        $orderItme = OrderItem::create($data);
        return $orderItme;
    }

    public function getOrder($income_firm_id, $sender_firm_id)
    {
        $order = Order::where('id_firm_income', $income_firm_id)->where('id_firm_sender', $sender_firm_id)
            ->where('status', OrderStatus::DRAFT)
            ->first();
        return $order;
    }

    public function getOrderList($order_id, $product_id, $price, $discount)
    {
        $orderList = OrderItem::where('order_id', $order_id)
            ->where('product_id', $product_id)
            ->where('price', $price)
            ->where('discount', $discount)
            ->first();
        return $orderList;
    }

    public function getOrderByOrderId($order_id, $firmId, $nowStatus)
    {
        $order = Order::where('id', $order_id)
            ->where('id_firm_sender', $firmId)
            ->where('status', $nowStatus)
            ->first();
        return $order;
    }

    public function updateStatus($firm_id, $order_id, $comment, $nowStatus, $newStatus)
    {
        Order::where('id', $order_id)
            ->where('status', $nowStatus)
            ->where('id_firm_sender', $firm_id)
            ->update([
                'status' => $newStatus,
                'approved' => 0,
                'comment' => $comment
            ]);

        return 'success';
    }


    public function deleteOrder($firmId, $orders_id)
    {
        Order::whereIn('id', $orders_id)
            ->update(['status' => OrderStatus::ARCHIVED]);
        return "deleteOrder";
    }

    public function getProductByItemId($orderList_id, $firmId)
    {
        $product = OrderItem::where('id', $orderList_id)->where('status', 0)->whereHas('order', function ($q) use ($firmId) {
            $q->where('id_firm_sender', $firmId)->where('status', OrderStatus::DRAFT);
        })->first();
        return $product;
    }

    public function deleteProductForOrderItem($orderList_id, $firmId)
    {
        OrderItem::where('id', $orderList_id)->whereHas('order', function ($q) use ($firmId) {
            $q->where('id_firm_sender', $firmId)->where('status', OrderStatus::DRAFT);
        })->update(
            [
                'status' => -1,
                'quantity' => 0
            ]
        );
        return 'deleted product';
    }

    public function updateQuantity($orderList_id, $firmId, $quantity)
    {
        $orderItem = OrderItem::where('id', $orderList_id)->whereHas('order', function ($q) use ($firmId) {
            $q->where('id_firm_sender', $firmId);
        })->first();

        if (!$orderItem) {
            return "empty product";
        }

        $orderItem->update(['quantity' => $quantity]);

        $orderId = $orderItem->order_id;

        $total_sum = OrderItem::where('order_id', $orderId)
            ->where('status', '>=', 0)
            ->sum(DB::raw('quantity * price'));

        $total_sum_with_discount = OrderItem::where('order_id', $orderId)
            ->where('status', '>=', 0)
            ->sum(DB::raw('quantity * price * (1 - discount / 100)'));

        return [
            'total_price' => number_format($total_sum, 2, '.', ''),
            'total_price_with_discount' => number_format($total_sum_with_discount, 2, '.', '')
        ];
    }

    public function getOrderItemByOrderId($firmId, $orderId)
    {
        $totlal_sum = OrderItem::select(DB::raw('SUM(quantity * price)'))
            ->whereColumn('order_id', 'orders.id')
            ->where('status', '!=', -1);

        $totlal_sum_with_discount = OrderItem::select(DB::raw('SUM(quantity * price*(1-discount/100))'))
            ->whereColumn('order_id', 'orders.id')
            ->where('status', '!=', -1);

        $client_name = Firm::select('title')
            ->whereColumn('id', 'orders.id_firm_income')
            ->limit(1);

        return Order::with(
            [
                'items' => function ($query) {
                    $query->where('status', '!=', -1);
                },
                'items.product.overhead',
                'supplier'
            ]
        )
            ->addSelect(['total_price' => $totlal_sum, 'totlal_sum_with_discount' => $totlal_sum_with_discount, 'client' => $client_name])
            ->where('id_firm_sender', $firmId)
            ->findOrFail($orderId);
    }

    public function getProductByOderItemId($firmId, $oderItemId)
    {
        return OrderItem::select('orderlists.id', 'orderlists.discount', 'orderlists.manufacture', 'orderlists.date_expire', 'orderlists.quantity', 'orderlists.price')
            ->join('orders', 'orderlists.order_id', '=', 'orders.id')
            ->where('orders.id_firm_sender', $firmId)
            ->find($oderItemId);
    }

    public function getProductQuantityById(int $product_id, int $supplier_id): ?int
    {
        return OrderItem::join('orders', 'orderlists.order_id', '=', 'orders.id')
            ->where('orders.id_firm_income', $supplier_id)
            ->where('orders.status', OrderStatus::DRAFT)
            ->where('orderlists.product_id', $product_id)
            ->value('orderlists.quantity');
    }
}
