<?php

namespace App\Repositories;

use App\Interfaces\PartnerInterface;
use App\Models\Firm;
use App\Models\Order;
use App\Models\OrderStatus;
use App\Models\Review;
use Carbon\Carbon;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;


class PartnerRepository implements PartnerInterface
{
    public function partners($firm_id, ?int $cityId = null, ?string $search = null): LengthAwarePaginator
    {
        $query = DB::table('partnergroups')
            ->select('firms.id', 'firms.img', 'firms.title', 'firms.id_city', 'cities.title as city', 'cities.country as country')
            ->join('firms', 'partnergroups.firm2', '=', 'firms.id')
            ->join('cities', 'firms.id_city', '=', 'cities.id')
            ->where('partnergroups.firm1', $firm_id);

        if ($search) {
            $query->where('firms.title', 'like', '%' . $search . '%');
        }
        if ($cityId) {
            $query->where('firms.id_city', $cityId);
        }

        return $query->paginate(10);
    }

    public function partnerInfo($firmId, $partnerId): Collection
    {
        return DB::table('partnergroups')
            ->select(
                'firms.id',
                'firms.title',
                'firms.img',
                'firms.phone',
                'firms.phone1',
                'firms.email',
                'firms.online_order',
                'cities.title as city_title',
                'cities.country as city_country',
                'firms.latitude as latitude',
                'firms.longtitude as longitude',
            )
            ->join('firms', 'partnergroups.firm2', '=', 'firms.id')
            ->join('cities', 'firms.id_city', '=', 'cities.id')
            ->leftJoin('overheads', 'firms.id', '=', 'overheads.firm_id')
            ->where('partnergroups.firm1', $firmId)
            ->where('partnergroups.firm2', $partnerId)
            ->limit(1)
            ->get();
    }

    public function statistic(array $filters)
    {
        $orderBaseQuery = Order::query()
            ->whereColumn('orders.id_firm_income', 'firms.id');

        if (!empty($filters['date_start'])) {
            $orderBaseQuery->whereDate(
                'orders.created_at',
                '>=',
                Carbon::createFromFormat('d-m-Y', $filters['date_start'])->format('Y-m-d')
            );
        }

        if (!empty($filters['date_end'])) {
            $orderBaseQuery->whereDate(
                'orders.created_at',
                '<=',
                Carbon::createFromFormat('d-m-Y', $filters['date_end'])->format('Y-m-d')
            );
        }

        $query = Firm::query()
            ->where('id', $filters['supplier_id'])
            ->select('id')
            ->addSelect([

                'order_in_progress' => (clone $orderBaseQuery)
                    ->selectRaw('count(*)')
                    ->where('orders.status', OrderStatus::SENT),

                'order_all_counts' => (clone $orderBaseQuery)
                    ->selectRaw('count(*)')
                    ->where('orders.status', OrderStatus::COMPLETED),

                'order_total_prices' => (clone $orderBaseQuery)
                    ->join('orderlists', 'orderlists.order_id', '=', 'orders.id')
                    ->selectRaw(
                        'SUM((orderlists.quantity * orderlists.price)*(1-(orders.discount)/100))'
                    )
                    ->where('orders.status', OrderStatus::COMPLETED),

                'dislike_count' => Review::query()
                    ->selectRaw('count(*)')
                    ->join('reason_lists', 'reason_lists.id', '=', 'reviews.reason_list_id')
                    ->whereColumn('reviews.supplier_id', 'firms.id')
                    ->where('reason_lists.type', 'dislike'),

                'average_order_value' => (clone $orderBaseQuery)
                    ->join('orderlists', 'orderlists.order_id', '=', 'orders.id')
                    ->selectRaw(
                        'AVG((orderlists.quantity * orderlists.price)*(1-(orders.discount)/100))'
                    )
                    ->where('orders.status', OrderStatus::COMPLETED),
            ]);

        return $query->first();

    }

    public function getCheckedPartnerIds($firmId, $checkedCityIds): Collection
    {
        $cacheKey = "firm_{$firmId}_my_partners_status";
        $cachedStatus = Cache::get($cacheKey, []);
        $unCheckedPartner = collect($cachedStatus)
            ->filter(fn($value) => $value === '0' || $value === false)
            ->keys()
            ->toArray();
        $query = Firm::query()
            ->join('partnergroups', 'partnergroups.firm2', '=', 'firms.id')
            ->where('partnergroups.firm1', $firmId)
            ->whereIn('firms.id_city', $checkedCityIds);
        if (!empty($unCheckedPartner)) {
            $query->whereNotIn('firms.id', $unCheckedPartner);
        }
        return $query->pluck('firms.id');
    }


    public function countChecked($firmId): int
    {
        $cacheKey = "firm_{$firmId}_my_partners_status";
        $cachedStatus = Cache::get($cacheKey, []);
        $unCheckedPartner = collect($cachedStatus)
            ->filter(fn($value) => $value === '0' || $value === false)
            ->keys()
            ->toArray();

        $query = DB::table('partnergroups')
            ->where('firm1', $firmId);

        if (!empty($unCheckedPartner)) {
            $query->whereNotIn('firm2', $unCheckedPartner);
        }

        return $query->count();
    }

}
