<?php

namespace App\Repositories;

use App\Interfaces\PaymentRepositoryInterface;
use App\Models\Enums\PaymentStatus;
use App\Models\Enums\TransactionStatus;
use App\Models\Firm;
use App\Models\FirmPlanStatus;
use App\Models\Payment;
use App\Models\PaymentMethod;
use App\Models\Plan;
use App\Models\Transaction;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class PaymentRepository implements PaymentRepositoryInterface
{
    public function getPayments(): Collection
    {
        return PaymentMethod::query()->select('id', 'title', 'text', 'image')->with(['paymentImages' => function ($query) {
            $query->select('payment_method_id', 'image');
        }])->where('status', 1)->get();
    }

    public function getPaymentId(int $firmId)
    {
        return Payment::join('firm_plans', 'payments.firm_plan_id', '=', 'firm_plans.id')
            ->leftJoin('transactions', 'transactions.payment_id', '=', 'payments.id')
            ->where('firm_plans.firm_id', $firmId)
            ->where('payments.status', PaymentStatus::NotPaid)
            ->where('transactions.exp_time', '>', now())
            ->select('payments.id', 'payments.amount')
            ->first();
    }

    public function create($firm_plan_id, $amount)
    {
        return Payment::create([
            'firm_plan_id' => $firm_plan_id,
            'amount' => $amount]);
    }

    public function getFirmPlan($firmId)
    {
        return DB::table('firm_plans')->select('firm_plans.id', 'plans.type', 'plans.price')
            ->join('plans', 'plans.id', '=', 'firm_plans.plan_id')
            ->where('firm_plans.firm_id', $firmId)
            ->whereNot('firm_plans.status', FirmPlanStatus::Inactive)
            ->first();
    }

    public function createTransaction($result, int $payment_id, int $method_id): void
    {
        dd($result);
        Transaction::create([
            'amount' => $result->amount,
            'payment_id' => $payment_id,
            'method_id' => $method_id,
            'invoice_id' => $result->invoice_id,
            'invoice_url' => $result->invoice_url,
            'exp_time' => $result->exp_time
        ]);
    }

    public function GetTransactionByPaymentId(int $payment_id): ?Transaction
    {
        return Transaction::where('payment_id', $payment_id)
            ->where('status', TransactionStatus::NotPaid)
            ->first();
    }

    public function updateTransactionById(int $id, array $date): string
    {
        Transaction::where('id', $id)
            ->update($date);
        return 'success';
    }

    public function getPlanTypeByPaymentId($payment_id): ?Plan
    {
        return Plan::select('plans.id', 'plans.type', 'plans.price')
            ->join('firm_plans', 'plans.id', '=', 'firm_plans.plan_id')
            ->join('payments', 'payments.firm_plan_id', '=', 'firm_plans.id')
            ->where('payments.status', PaymentStatus::NotPaid)
            ->where('payments.id', $payment_id)
            ->first();
    }

    public function getPaymentById($payment_id): Payment
    {
        return Payment::where('status', PaymentStatus::NotPaid)
            ->where('id', $payment_id)
            ->first();
    }

    public function getAmountTotalByPaymentId($payment_id)
    {
        $total = Payment::join('transactions', 'transactions.payment_id', '=', 'payments.id')
            ->where('transactions.status', TransactionStatus::Paid)
            ->where('payments.id', $payment_id)
            ->selectRaw('SUM(transactions.amount) as total_amount')
            ->value('total_amount');
        return $total ?? 0;
    }

    public function getFirmIdByFirmPlanId($firmPlanId)
    {
        return DB::table('firm_plans')
            ->where('id', $firmPlanId)
            ->value('firm_id');
    }

    public function updateFirmPlanByFirmId(int $firmId, array $data): void
    {
        DB::table('firm_plans')
            ->where('firm_id', $firmId)
            ->update($data);
    }

    public function updateFirmLeftDaysByFirmId(int $firmId, int $amount): void
    {
        Firm::where('id', $firmId)
            ->update(['left_days' => $amount]);
    }

    public function updatePaymentStatusById($paymentId): void
    {
        Payment::where('id', $paymentId)
            ->update(['status', PaymentStatus::Paid]);
    }

}
