<?php

namespace App\Services;

use App\Interfaces\Cities\CityInterface;
use App\Models\City;
use Illuminate\Support\Facades\Cache;

class CityService
{
    public function __construct
    (
        protected CityInterface $cityInterface
    )
    {
    }

    /**
     * @OA\Get(
     *      path="/api/getCities",
     *      tags={"City"},
     *      summary="Get all cities",
     *      description="This returns all available cities",
     *      @OA\Response(
     *          response=200,
     *          description="succsses",
     *            @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *       ),
     *     )
     */
    public function getCities($firmId)
    {

        $cities = $this->cityInterface->getCity();

        if ($firmId) {
            $cacheKey = "firm_{$firmId}_cities_status";
            $cachedStatus = Cache::get($cacheKey, []);

            foreach ($cities as &$city) {
                $city->checked = $cachedStatus[$city->id] ?? true;
            }
        }

        return $cities;
    }

    /**
     * @OA\Get(
     *      path="/api/city/by_checked",
     *      tags={"City"},
     *       security={{"bearerAuth":{}}},
     *      summary="",
     *      description="",
     *     @OA\Parameter(
     *           name="firm_id",
     *           in="header",
     *           required=true,
     *           @OA\Schema(
     *               type="integer",
     *               example=1
     *           )
     *       ),
     *      @OA\Response(
     *          response=200,
     *          description="Successful operation",
     *          @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *      ),
     *  )
     */
    public function citiesByChecked($request)
    {
        $firmId = $request->header('firm_id');
        $cacheKey = "firm_{$firmId}_cities_status";
        $cachedStatus = Cache::get($cacheKey, []);
        $cities = $this->cityInterface->getCity();

        foreach ($cities as &$city) {
            $city->checked = $cachedStatus[$city->id] ?? true;
        }

        return $cities;
    }

    /**
     * @OA\Post(
     *     path="/api/city/update_checked",
     *     tags={"City"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="city_id",
     *                     type="integer",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="checked",
     *                     type="boolean",
     *                     example=true
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function updateChecked($request): true
    {
        $firmId = $request->header('firm_id');
        $cityId = $request->city_id;
        $checked = $request->checked;

        $cacheKey = "firm_{$firmId}_cities_status";
        $cachedStatus = Cache::get($cacheKey, []);

        $cachedStatus[$cityId] = $checked;

        Cache::put($cacheKey, $cachedStatus);
        return true;
    }
}
