<?php

namespace App\Services\Elasticsearch;

use App\Models\Drugstore;
use App\Models\Review;
use App\Services\Drugstore\DiscountService;


class ElasticsearchService
{
    public function __construct
    (
        protected DiscountService $discountService
    )
    {
    }


    public function createIndex()
    {
        $client = app('elasticsearch');
        $params = [
            'index' => 'drugstore',
            'body' => [
                'settings' => [
                    'analysis' => [
                        'filter' => [
                            'edge_ngram_filter' => [
                                'type' => 'edge_ngram',
                                'min_gram' => 3,
                                'max_gram' => 20
                            ],
                            "apostrophe_filter" => [
                                "type" => "apostrophe"
                            ]
                        ],
                        'analyzer' => [
                            'edge_ngram_analyzer' => [
                                'tokenizer' => 'standard',
                                'filter' => ['lowercase', 'edge_ngram_filter']
                            ],
                            'standard_analyzer' => [
                                'tokenizer' => 'standard',
                                'filter' => ['lowercase', 'apostrophe_filter']
                            ]
                        ]
                    ]
                ],
                'mappings' => [
                    'properties' => [
                        'price' => ['type' => 'float'],
                        'overhead_id' => ['type' => 'integer'],
                        'title' => [
                            'type' => 'text',
                            'analyzer' => 'standard_analyzer',
                            'fields' => [
                                'prefix' => ['type' => 'text', 'analyzer' => 'edge_ngram_analyzer'],
                                'keyword' => ['type' => 'keyword', 'ignore_above' => 256]
                            ]
                        ],
                    ]
                ]
            ]
        ];

        return $client->indices()->create($params)->asArray();
    }

    public function synced()
    {
        $client = app('elasticsearch');
        foreach (Drugstore::with('overhead.firm', 'overhead.discount')->cursor() as $p) {
            $discount = $this->discountService->discount($p) ?? 0;
            $ratings = Review::with('reasonList')
                ->get()
                ->groupBy('product_id')
                ->map(function ($reviews) {
                    $likes = $reviews->where('reasonList.type', 'like')->count();
                    $dislikes = $reviews->where('reasonList.type', 'dislike')->count();
                    $total = $likes + $dislikes;

                    if ($total === 0) return 1;
                    $rating = round(($likes / $total) * 5);
                    return max(0, min(5, $rating));
                });

            $client->index([
                'index' => 'drugstore',
                'id' => $p->id,
                'body' => [
                    'id' => $p->id,
                    'title' => $p->title,
                    'product_img' => $p->img ?? 'products/1.png',
                    'discounts' => $discount,
                    'price' => $p->price,
                    'price_with_discount' => number_format(($p->price) * (1 - $discount / 100), 2),
                    'manufacture' => $p->manufacture,
                    'date_expire' => $p->date_expire,
                    'date_price_list' => $p->overhead->date_overhead ?? null,
                    'overhead_id' => $p->overhead_id,
                    'firm_id' => $p->overhead->firm->id ?? null,
                    'firm_title' => $p->overhead->firm->title ?? null,
                    'rating' => $ratings[$p->id] ?? 0,
                ],
            ]);
        }

        return ['message' => 'Drugstore synced'];
    }

    public function destroy($request)
    {
        $id = $request->id;
        $client = app('elasticsearch');
        $result = $client->delete(['index' => 'drugstore', 'id' => $id]);
        return $result;
    }

    public function deleteindex()
    {
        app('elasticsearch')->indices()->delete(['index' => 'drugstore']);
        return 'success';
    }

}
