<?php

namespace App\Services;

use App\Interfaces\FirmInterface;
use App\Interfaces\FirmPlans\FirmPlanInterface;
use App\Models\Firm;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;


class FirmService
{
    /**
     * @OA\Post(
     *     path="/api/createfirm",
     *     tags={"Firm"},
     *     security={{"bearerAuth":{}}},
     *     @OA\RequestBody(
     *         required=true,
     *     @OA\MediaType(
     *              mediaType="multipart/form-data",
     *             @OA\Schema(
     *                required={"title", "id_category", "id_city"},
     *                @OA\Property(
     *                      property="title",
     *                      type="string",
     *                      example="New Firm"
     *                  ),
     *                  @OA\Property(
     *                      property="id_category",
     *                      type="integer",
     *                      example=1
     *                  ),
     *                  @OA\Property(
     *                      property="id_city",
     *                      type="integer",
     *                      example=2
     *                  ),
     *                  @OA\Property(
     *                      property="photo",
     *                      type="string",
     *                      format="binary"
     *                  )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function __construct
    (
        protected FirmInterface      $firmInterface,
        protected FormatPhoneService $formatPhoneService,
        protected FirmPlanInterface  $firmPlanInterface
    )
    {
    }

    public function createFirm($request)
    {
        $user = Auth::user();

        $data = [
            'title' => $request->title,
            'id_category' => 1, // default category: Drugstore
            'id_city' => $request->id_city,
            'phone' => $user->phone,
        ];
        if ($request->hasFile('photo')) {
            $file = $request->file('photo');
            $filename = Str::uuid() . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('firms', $filename, 'public');
            $data['img'] = $path;
        } else {
            $data['img'] = 'firms/log.png';
        }
        $firm = $this->firmInterface->createFirm($data);

        $user->firms()->attach($firm->id);


        $tariff = Cache::put($firm->id, 'Тариф не выбран');
        $firm['tariff'] = Cache::get($firm->id);

        return $firm;
    }

    public function myFirms()
    {
        $user = Auth::user();
        $firms = Firm::select('firms.id', 'firms.img', 'firms.title', 'firms.phone', 'firms.left_days', 'firms.status', 'firms.date_sign', 'categories.title as category_title')
            ->join('userfirms', 'firms.id', '=', 'userfirms.firm_id')
            ->join('categories', 'firms.id_category', '=', 'categories.id')
            ->where('userfirms.user_id', $user->id)
            ->whereIn('userfirms.status', [0, 1])
            ->get();
        $firms = $firms->map(function ($firm) {
            $tariffPlan = $this->firmPlanInterface->getFirmPlan($firm->id);
            if ($tariffPlan) {
                $firm->tariff_plan_id = $tariffPlan->id;
                $firm->tariff_plan = $tariffPlan->name;
                $firm->status = $tariffPlan->status;
            } else {
                $firm->tariff_plan_id = null;
                $firm->tariff_plan = "Тариф не выбран";
                if ($firm->status == 0) {
                    $firm->status = "moderation";
                } else {
                    $firm->status = "no_plan";
                }
            }
            return $firm;
        });

        if (!$firms) {
            return $user->phone;
        }
        return $firms;

    }

    /**
     * @OA\Get(
     *      path="/api/firm_categories",
     *      tags={"Firm"},
     *       security={{"bearerAuth":{}}},
     *      summary="Get  firm categories",
     *      description="Get  firm categories",
     *      @OA\Response(
     *          response=200,
     *          description="Successful operation",
     *          @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *      ),
     *  )
     */
    public function categoriesFirm()
    {
        return $this->firmInterface->categoriesFirm();
    }

    public function firmProfile($request)
    {
        $firmId = $request->header('firm_id');

        return $this->firmInterface->firmProfile($firmId);
    }

    public function workingData($request)
    {
        $firmId = $request->header('firm_id');

        return $this->firmInterface->workingData($firmId);
    }

    public function updateWorkingData($request): string
    {
        $firmId = $request->header('firm_id');
        $firm = $this->firmInterface->getFirmByID($firmId);

        $data = [
            'address' => $request->address,
            'phone' => $this->formatPhoneService
                ->formatPhone($request->country_code, $request->phone),
        ];

        if ($request->additional_phone)
            $data['phone1'] = $this->formatPhoneService->formatPhone($request->additional_country_code, $request->additional_phone);

        if ($request->HasFile('photo')) {
            if ($firm->img && Storage::disk('public')->exists($firm->img)) {
                Storage::disk('public')->delete($firm->img);
            }
            $file = $request->file('photo');
            $filename = Str::uuid() . '.' . $file->getClientOriginalExtension();
            $path = $file->storeAs('firms', $filename, 'public');
            $data['img'] = $path;
        }

        $updated = $this->firmInterface->updateWorkingData($data, $firmId);

        return $updated ? 'Updated successfully' : 'error in updated';
    }

    public function updateGeo($request)
    {
        $firmId = $request->header('firm_id');
        $data = [
            'latitude' => $request->latitude,
            'longtitude' => $request->longitude,
        ];
        $updated = $this->firmInterface->updateWorkingData($data, $firmId);

        if ($updated) {
            return $this->firmInterface->workingData($firmId);
        }

        return 'Error in update';
    }

}
