<?php

namespace App\Services\Nomenclatures;

use App\Interfaces\Nomenclatures\NomenclatureInterface;
use App\Models\Nomenclature;
use Illuminate\Pagination\LengthAwarePaginator;

class NomenclatureService
{
    public function __construct
    (
        protected NomenclatureInterface $nomenclatureInterface,
    )
    {
    }

    /**
     * @OA\Post(
     *     path="/api/nomenclatures/get_nomenclatures_by_search",
     *     tags={"Nomenclatures"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="search_query",
     *                     type="string",
     *                     example="Параце"
     *                 ),
     *                 @OA\Property(
     *                     property="product_type_id",
     *                     type="integer",
     *                     example=1
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function get_nomenclatures_by_search(string $search_query, int $product_type_id): LengthAwarePaginator
    {
        $results = $this->nomenclatureInterface->get_nomenclatures_by_search($search_query, $product_type_id);
        $results->getCollection()->transform(function ($data) {
            return [
                'nomenclature_id' => $data->id,
                'nomenclature_name' => $data->name,
                'nomenclature_dose' => $data->dose,
                'nomenclature_article' => str_pad((string)$data->id, 4, '0', STR_PAD_LEFT),
                'nomenclature_alias' => $data->alias,
                'nomenclature_mnn' => $data->mnn,

                'images' => $data->images->map(function ($img) {
                    return [
                        'image_url' => $img->image_url,
                        'is_main' => $img->is_main,
                    ];
                }),
                'type' => $data->type?->name,
                'manufacturer_name' => $data->manufacturer?->name,
            ];
        });
        return $results;
    }

    /**
     * @OA\Post(
     *     path="/api/nomenclatures/get_details_by_id",
     *     tags={"Nomenclatures"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="nomenclature_id",
     *                     type="integer",
     *                     example=1
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function get_details_by_id($nomenclature_id): Nomenclature
    {
        return $this->nomenclatureInterface->get_details_by_id($nomenclature_id);
    }

    /**
     * @OA\Post(
     *     path="/api/nomenclatures/get_analogues",
     *     tags={"Nomenclatures"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="search_query",
     *                     type="string",
     *                     example="Параце"
     *                 ),
     *                 @OA\Property(
     *                     property="product_type_id",
     *                     type="integer",
     *                     example=1
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function get_analogues($nomenclature_id): LengthAwarePaginator
    {
        $results = $this->nomenclatureInterface->get_analogues($nomenclature_id);

       $results->getCollection()->transform(function ($data) {
            return [
                'nomenclature_id' => $data->id,
                'nomenclature_name' => $data->name,
                'nomenclature_dose' => $data->dose,
                'nomenclature_article' => str_pad((string)$data->id, 4, '0', STR_PAD_LEFT),
                'nomenclature_alias' => $data->alias,
                'nomenclature_mnn' => $data->mnn,

                'images' => $data->images->map(function ($img) {
                    return [
                        'image_url' => $img->image_url,
                        'is_main' => $img->is_main,
                    ];
                }),
                'type' => $data->type?->name,
                'manufacturer_name' => $data->manufacturer?->name,
            ];
        });
        return  $results;
    }


}
