<?php

namespace App\Services\OrderServices;

use App\Interfaces\OrderInterfaces\OrderInterface;
use App\Interfaces\Products\ProductInterface;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\OrderStatus;
use App\Models\Overhead;
use App\Services\Drugstore\DiscountService;
use App\Services\PartnerService;


class OrderService
{
    public function __construct
    (
        protected OrderInterface   $orderInterface,
        protected PartnerService   $partnerService,
        protected ProductInterface $product,
        protected DiscountService  $discountService
    )
    {
    }

    /**
     * @OA\Post(
     *     path="/api/orders/addtoorder",
     *     tags={"Orders"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="product_id",
     *                     type="integer",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="quantity",
     *                     type="integer",
     *                     example=1
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function addToOrder($request): string
    {
        $sender_firm_id = (int)$request->header('firm_id');
        $product_id = $request->product_id;
        $quantity = $request->quantity;
        $product = $this->product->getProductById($product_id, $sender_firm_id);

        if (!$product) {
            return 'Product not available or not authorized.';
        }
        $overhead_id = $product->overhead_id;
        $title = $product->title;

        $income_firm_id = Overhead::where('id', $overhead_id)->value('firm_id');
        $price = $product->price;
        $date_expire = $product->date_expire;
        $manufacture = $product->manufacture;
        $discount = $this->discountService->discount($product);

        $order = $this->orderInterface->getOrder($income_firm_id, $sender_firm_id);
        $length = max(6, strlen((string) Order::max('number') + 1));
        if (!$order) {
            $data = [
                'overhead_id' => $overhead_id,
                'id_firm_sender' => $sender_firm_id,
                'id_firm_income' => $income_firm_id,
                'number' => str_pad(Order::max('number') + 1, $length, '0', STR_PAD_LEFT),
                'comment' => $request->comment ?? '',
                'discount' => $discount
            ];

            $order = $this->orderInterface->addToOrder($data);
        }
        $orderList = $this->orderInterface->getOrderList($order->id, $product_id, $price, $discount);

        if ($orderList) {
            OrderItem::where('product_id', $product_id)->update(
                [
                    'quantity' => $quantity,
                    'status' => 0
                ]
            );
        } else {
            $orderList = [
                'order_id' => $order->id,
                'product_id' => $product_id,
                'title' => $title,
                'quantity' => $quantity,
                'price' => $price,
                'discount' => $discount,
                'date_expire' => $date_expire,
                'manufacture' => $manufacture,
                'comment' => $request->comment ?? '',
            ];
            $this->orderInterface->addToOrderItem($orderList);
        }

        return 'success';
    }

    /**
     * @OA\Get(
     *     path="/api/orders/orders",
     *     tags={"Orders"},
     *     security={{"bearerAuth":{}}},
     *     summary="Get all orders",
     *
     *     @OA\Parameter(
     *         name="firm_id",
     *         in="query",
     *         required=true,
     *         @OA\Schema(
     *             type="integer",
     *             example=1
     *         )
     *     ),
     *
     *     @OA\Response(
     *         response=200,
     *         description="Successful operation",
     *         @OA\MediaType(
     *             mediaType="application/json"
     *         )
     *     )
     * )
     */

    public function orders($request)
    {
        $firmId = $request->header('firm_id');
        $dateStart = $request->date_start ?? null;
        $dateEnd = $request->date_end ?? null;
        $status = $request->status ?? null;
        $partnerId = $request->partner_id ?? null;

        if ($partnerId) {
            $partnerIds = array($partnerId);
        } else {
            $partners = $this->partnerService->partners($request);
            $partnerIds = $partners->where('status', true)->pluck('id')->toArray();
        }
        $search = $request->search ?? null;
        return $this->orderInterface->order($firmId, $partnerIds, $search, $dateStart, $dateEnd, $status);
    }

    /**
     * @OA\Post(
     *     path="/api/orders/send_order",
     *     tags={"Orders"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="order_id",
     *                     type="integer",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="comment",
     *                     type="string",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function SendOrder($request): string
    {
        $firmId = $request->header('firm_id');
        $order_id = $request->order_id;
        $comment = $request->comment ?? '';
        $nowstatus = OrderStatus::DRAFT;
        $newstatus = OrderStatus::SENT;

        $data = $this->orderInterface->getOrderByOrderId($order_id, $firmId, $nowstatus);

        if (empty($data)) {
            return 'note found order';
        }
        $this->orderInterface->updateStatus($firmId, $order_id, $comment, $nowstatus, $newstatus);

        return 'success';
    }

    /**
     * @OA\Post(
     *     path="/api/orders/restore_from_history",
     *     tags={"Orders"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="order_id",
     *                     type="integer",
     *                     example=1
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function deleteOrder($request)
    {
        $firmId = $request->header('firm_id');
        $orders_id = $request->ids;
        return $this->orderInterface->deleteOrder($firmId, $orders_id);
    }

    /**
     * @OA\Get(
     *      path="/api/orders/save_order_all_history",
     *      tags={"Orders"},
     *       security={{"bearerAuth":{}}},
     *      summary="save orders all history",
     *      description="save_order_all_history",
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *      @OA\Response(
     *          response=200,
     *          description="Successful operation",
     *          @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *      ),
     *  )
     */
    public function deleteProductForOrderItem($request)
    {
        $firmId = $request->header('firm_id');
        $orderList_id = $request->id;
        $product = $this->orderInterface->getProductByItemId($orderList_id, $firmId);
        if (empty($product)) {
            return 'empty product';
        }
        return $this->orderInterface->deleteProductForOrderItem($orderList_id, $firmId);

    }

    /**
     * @OA\Post(
     *     path="/api/orders/update_quantity",
     *     tags={"Orders"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="id",
     *                     type="integer",
     *                     example=1
     *                 ),
     *                 @OA\Property(
     *                     property="quantity",
     *                     type="integer",
     *                     example=1
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function updateQuantity($request)
    {
        $firmId = $request->header('firm_id');
        $orderList_id = $request->id;
        $quantity = $request->quantity;
        $product = $this->orderInterface->getProductByItemId($orderList_id, $firmId);
        if (empty($product)) {
            return 'empty product';
        }
        return $this->orderInterface->updateQuantity($orderList_id, $firmId, $quantity);
    }

    public function getStatus(): array
    {

        $statuses = [];
        foreach (Order::getStatuses() as $id => $status) {
            $statuses[] = [
                'id' => $id,
                'status' => $status,
            ];
        }
        return $statuses;
    }

    public function getOrderItemsByOrderId($request): array
    {
        $firmId = $request->header('firm_id');
        $orderId = $request->order_id;

        $order = $this->orderInterface->getOrderItemByOrderId($firmId, $orderId);
        if (!$order) return [];

        if ($order->supplier) {
            $supplier = [
                'id' => $order->supplier->id,
                'name' => $order->supplier->title,
            ];
        } else {
            $supplier = null;
        };

        return [
            'id' => $order->id,
            'number' => $order->number,
            "supplier" => $supplier,
            "status" => $order->status,
            "total_price" => number_format($order->total_price, 2),
            "total_price_with_discount" => number_format($order->totlal_sum_with_discount, 2),
            "date_time" => $order->created_at->format('d.m.Y - H:i:s'),
            "client" => $order->client,
            "items" => $order->items->map(function ($item) {
                return [
                    "id" => $item->id,
                    "photo" => $item->photo ?? 'products/1.png',
                    "name" => $item->title,
                    "manufacture" => $item->manufacture,
                    "date_expire" => $item->date_expire,
                    "quantity" => $item->quantity,
                    "price" => $item->price,
                    "discount" => number_format($item->discount, 2),
                    "price_with_discount" => number_format($item->price * (1 - $item->discount / 100), 2),
                    "upload_date" => $item->product->overhead->date_overhead ?? null,
                ];
            }),
        ];
    }

    public function getProductByOderItemId($request)
    {
        $firmId = $request->header('firm_id');
        $orderItemId = $request->orderitem_id;
        $product = $this->orderInterface->getProductByOderItemId($firmId, $orderItemId);

        if (!$product) {
            return [];
        }

        $data = [
            'id' => $product->id,
            'product_img' => $product->img ?? 'products/1.png',
            'discount' => $product->discount,
            'price' => $product->price,
            'price_with_discount' => number_format(($product->price) * (1 - $product->discount / 100), 2),
            'manufacture' => $product->manufacture,
            'date_expire' => $product->date_expire,
            'upload_date    ' => $product->date_expire,
            'quantity' => $product->quantity
        ];

        return $data;
    }

}
