<?php

namespace App\Services;

use App\Interfaces\FirmInterface;
use App\Interfaces\PartnerInterface;
use Illuminate\Support\Facades\Cache;

class PartnerService
{
    public function __construct
    (
        protected PartnerInterface $partner_interface,
        protected FirmInterface    $firm_interface
    )
    {
    }

    /**
     * @OA\Get(
     *      path="/api/my_partners",
     *      tags={"Partners"},
     *       security={{"bearerAuth":{}}},
     *      summary="Get my Partners",
     *      description="Get my Partners",
     *       @OA\Parameter(
     *         name="firm_id",
     *         in="header",
     *         required=true,
     *         @OA\Schema(
     *             type="integer",
     *             example=1
     *         )
     *     ),
     *      @OA\Response(
     *          response=200,
     *          description="Successful operation",
     *          @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *      ),
     *  )
     */
    public function partners($request)
    {
        $firmId = $request->header('firm_id');
        $cityId = $request->city_id ?? null;
        $search = $request->search ?? null;
        $partners = $this->partner_interface->partners($firmId, $cityId, $search);

        $cacheKey = "firm_{$firmId}_my_partners_status";
        $cachedStatus = Cache::get($cacheKey, []);

        $partners->getCollection()->transform(function ($partner) use ($cachedStatus) {
            $partner->status = $cachedStatus[$partner->id] ?? true;
            return $partner;
        });

        return $partners;
    }

    public function partnerInfo($request)
    {
        $firmId = $request->header('firm_id');
        $partnerId = $request->partner_id;

        return $this->partner_interface->partnerInfo($firmId, $partnerId);
    }

    public function updateChecked($request): bool
    {
        $currentFirmId = $request->header('firm_id');
        $targetFirmId = $request->target_firm_id;
        $status = $request->status;

        $cacheKey = "firm_{$currentFirmId}_my_partners_status";
        $cachedStatus = Cache::get($cacheKey, []);

        $cachedStatus[$targetFirmId] = $status;

        Cache::put($cacheKey, $cachedStatus);
        return true;
    }

    public function statistic( array $filters): array
    {
        $data = $this->partner_interface->statistic($filters);
      if (!$data){
          return  [];
      }
        return [
            'firm_id' => $data->id,
            'order_in_progress' => $data->order_in_progress,
            'dislike_count' => $data->dislike_count,
            'order_all_counts' => $data->order_all_counts,
            'order_total_prices' => number_format($data->order_total_prices, 2),
            'changes_in_product_prices' => $data->changes_in_product_prices ?? 0,
            'average_order_value' => number_format($data->average_order_value, 2) ?? 0,
            'average_time_to_close_applications' => $data->average_time_to_close_applications ?? 0,
        ];
    }

    public function countChecked($request)
    {
        $firmId = $request->header('firm_id');
        return $this->partner_interface->countChecked($firmId);
    }
}
