<?php

namespace App\Services\Payments;

use App\Interfaces\PaymentRepositoryInterface;
use App\Models\Enums\PaymentMethods;
use App\Models\Enums\TransactionStatus;
use App\Models\Payment;
use App\Models\Transaction;


class PaymentService
{
    public function __construct(
        protected PaymentRepositoryInterface $payment,
        protected DcPaymentService           $dcPaymentService,
        protected AlifPaymentService         $alifPaymentService
    )
    {
    }

    /**
     * @OA\Get(
     *      path="/api/payment_info",
     *      tags={"Payment"},
     *       security={{"bearerAuth":{}}},
     *
     *      @OA\Response(
     *          response=200,
     *          description="Successful operation",
     *          @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *      ),
     *  )
     */

    public function getPayments(): Payment
    {
        return $this->payment->getPayments();
    }


    public function newPayments(int $firmId, $method_id)
    {
        $payments = $this->getPaymentId($firmId);

        if ($transaction = $this->payment->GetTransactionByPaymentId($payments->id)) {
            return $transaction;
        }

        $paymentService = [
            PaymentMethods::AlifMobi->value => $this->alifPaymentService,
            PaymentMethods::DcNext->value => $this->dcPaymentService,
        ];

        if (!isset($paymentService[$method_id])) {
            return "Not fount payment method";
        }

        $response = $paymentService[$method_id]->payments($payments);

        return $this->payment->createTransaction(
            $response->json(),
            $payments->id,
            $method_id
        );
    }

    public function getPaymentId(int $firmId)
    {
        $paymentId = $this->payment->getPaymentId($firmId);

        if (!$paymentId) {
            $paymentId = $this->create($firmId);
        }
        return $paymentId;
    }

    public function create(int $firmId)
    {
        $firm_plan_details = $this->payment->getFirmPlan($firmId);
        $firm_plan_id = $firm_plan_details->id;

        if ($firm_plan_details->type == 'month') {
            $amount = $firm_plan_details->price;
        } else {
            $amount = 1000;
        }
        return $this->payment->create($firm_plan_id, $amount);
    }

    /**
     * @throws \Exception
     */
    public function callbackDc(array $params): false|string
    {

        if (!$params['merchant'] == config('services.paymentDc.merchant')) {
            return false;
        }
//        $check = $this->dcPaymentService->check($params['order_id'], $params['invoice_id']);
//        if (!$check) {
//            return false;
//        }
//
//        preg_match('/<g>([\d.]+)<\/g>/', $check, $m);
//        $amountCheck = $m[1];
//
//        if (!$params['amount'] == $amountCheck) {
//            return false;
//        }
        $this->updateStatusTransaction($params);

        return 'success';
    }

    public function updateStatusTransaction(array $params): string
    {
        $transaction = $this->payment->GetTransactionByPaymentId($params['order_id']);

        if (!$transaction) {
            throw new  \Exception("Not found order");
        }

        $data = [
            'amount' => $params['amount'],
            'status' => TransactionStatus::Paid->value,
            'exp_time' => $params['pay_date'],
        ];

        $this->payment->updateTransactionById($transaction->id, $data);
        $this->updatePayments($params);

        return 'success';
    }

    public function updatePayments(array $params): void
    {
        $payments = $this->payment->getPaymentById($params['order_id']);
        $amountTotal = $this->payment->getAmountTotalByPaymentId($params['order_id']);
        if ($amountTotal >= $payments->amount) {
            $this->payment->updatePaymentStatusById($params['order_id']);
        }
    }

}
