<?php

namespace App\Services\Reviews;

use App\Interfaces\Products\ProductInterface;
use App\Interfaces\Reviews\ReviewInterface;
use App\Repositories\Products\ProductRepository;
use Illuminate\Pagination\LengthAwarePaginator;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;

class ReviewService
{
    public function __construct
    (
        protected ReviewInterface         $reviewInterface,
        protected ProductInterface        $product,
        protected ReviewStatisticsService $statisticsService
    )
    {
    }

    public function create($request): true
    {
        $product_id = $request->product_id;
        $comment = $request->comment;

        $data = [
            'supplier_id' => $request->supplier_id,
            'firm_id' => $request->header('firm_id'),
            'reason_list_id' => $request->reason_list_id,
            'product_name' => $this->product->getProductName($product_id),
            'comment' => $comment ?? '',
        ];

        $this->reviewInterface->create($data);
        return true;
    }

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function getAllReviews(int $supplier_id, int $page): array
    {
        $perPage = 10;
        $page = request()->get('page', 1);

        $ratingSummary = $this->statisticsService->getRatingSummary($supplier_id);
        $reviews = $this->statisticsService->getReviews($supplier_id);

        $paginated = new LengthAwarePaginator(
            $reviews->forPage($page, $perPage),
            $reviews->count(),
            $perPage,
            $page,
            ['path' => request()->url(), 'query' => request()->query()]
        );
        return [
            'summary' => $ratingSummary,
            'reviews' => $paginated
        ];
    }

    public function reasonList()
    {
        return $this->reviewInterface->get();
    }

}
