<?php

namespace App\Services\Reviews;

use App\Interfaces\Reviews\ReviewInterface;


class ReviewStatisticsService
{

    public function __construct
    (
        protected ReviewInterface $review,
    )
    {
    }

    public function getRatingSummary($supplier_id): array
    {
        $reviews = $this->review->getRating($supplier_id);

        $total = $reviews->total_count;
        $like = $reviews->positive_count;
        $dislike = $reviews->negative_count;

        if ($total > 0) {
            $likePercent = ($like / $total) * 100;
            $dislikePercent = ($dislike / $total) * 100;
            $rating = ($like / $total) * 5;
        } else {
            return [
                'rating' => 0,
                'total' => 0,
                'like_percent' => 0,
                'dislike_percent' => 0
            ];
        }

        return [
            'rating' => number_format($rating, 2),
            'total' => $total,
            'like_percent' => number_format($likePercent, 2),
            'dislike_percent' => number_format($dislikePercent, 2)
        ];
    }

    public function getReviews($supplier_id)
    {
       return $this->review->getReviews($supplier_id)
            ->groupBy(function ($item) {
                return $item->product_name . '-' . $item->reasonList->name;
            })
            ->map(function ($group) {
                $first = $group->first();

                return [
                    'product_name' => $first->product_name,
                    'reason_name' => $first->reasonList->name,
                    'reason_type' => $first->reasonList->type,
                    'reviews_count' => $group->count(),
                    'last_review_at' => $group->max('created_at'),
                    'firms' => $group->map(function ($review) {
                        return [
                            'firm_name' => $review->firm->title ?? null,
                            'firm_img' => $review->firm->img ?? null,
                            'comment'=>$review->comment ?? null,
                            'created_at'=>$review->created_at ,
                        ];
                    })->unique('firm_name')->values()
                ];
            })
            ->values();
    }


}
