<?php

namespace App\Services\Search;


use App\Interfaces\Cities\CityInterface;
use App\Interfaces\Overhead\OverheadInterface;
use App\Interfaces\PartnerInterface;
use App\Services\Search\QueryBuilder\FilterBuilder;
use App\Services\Search\QueryBuilder\ShouldQueryBuilder;
use App\Services\Search\QueryBuilder\SortBuilder;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection;

class ProductSearchService
{
    public function __construct
    (
        protected FilterBuilder        $filterBuilder,
        protected ShouldQueryBuilder   $shouldQueryBuilder,
        protected SortBuilder          $sortBuilder,
        protected ProductTypeService   $productTypeService,
        protected ElasticsearchService $elasticsearchService,
        protected OverheadInterface    $overheadInterface

    )
    {
    }

    public function search(array $params, int $firmId): LengthAwarePaginator|array
    {
        if (!empty($params['supplier_id'])) {
            return $this->searchProductsBySupplierId($params);
        } else {
            return $this->searchProductsByQuery($params, $firmId);
        }
    }

    public function searchProductsByQuery($params, $firmId): LengthAwarePaginator|array
    {
        $search_query = mb_strtolower(trim($params['search_query']));
        if (mb_strlen($search_query) < 3) {
            return [];
        } else {
            $partnerIds = $this->getPartnersIds($firmId);
            return $this->proceedSearch($params, $search_query, $partnerIds);
        }
    }

    public function searchProductsBySupplierId($params): LengthAwarePaginator|array
    {
        $search_query = mb_strtolower(trim($params['search_query']));
        $searchLength = mb_strlen($search_query);
        if ($searchLength < 3 and $searchLength > 0) {
            return [];
        } else {
            $partnerIds = collect($params['supplier_id']);
            return $this->proceedSearch($params, $search_query, $partnerIds);
        }
    }

    /**
     * @param array $params
     * @param string $search_query
     * @param Collection $partnerIds
     * @return array|LengthAwarePaginator
     */
    public function proceedSearch(array      $params,
                                  string     $search_query,
                                  Collection $partnerIds): array|LengthAwarePaginator
    {
        $from = ($params['page'] - 1) * 10;
        $words = explode(' ', trim($search_query));

        $productType = $this->productTypeService->typeById($params['product_type_id']);

        $overheadIds = $this->getOverheadIds($partnerIds);
        $filters = $this->filterBuilder->build($params, $overheadIds,$productType);
        $should = $this->shouldQueryBuilder->build($words);
        $sort = $this->sortBuilder->build($params['sort'] ?? null);

        return $this->elasticsearchService->search
        (
            index: 'drugstore',
            from: $from,
            size: $params['per_page'] ?? 10,
            filters: $filters,
            should: $should,
            sort: $sort
        );
    }

    private function getOverheadIds(Collection $partnerIds): Collection
    {
        return $this->overheadInterface->getOverheadIds($partnerIds);
    }

    private function getPartnersIds(int $firmId): Collection
    {
        $checkedCities = app(CityInterface::class)
            ->getCheckedCityIds($firmId);
        return app(PartnerInterface::class)
            ->getCheckedPartnerIds($firmId, $checkedCities);
    }

}
