<?php

namespace App\Services\Search;

use App\Interfaces\Cities\CityInterface;
use App\Interfaces\PartnerInterface;
use App\Models\Overhead;
use Illuminate\Pagination\LengthAwarePaginator;


class SearchService
{
    public function __construct
    (
        protected ProductTypeService $productTypeService,
        protected PartnerInterface $partnerInterface,
        protected CityInterface      $cityInterface,
    )
    {
    }

    /**
     * @OA\Post(
     *     path="/api/search/search",
     *     tags={"Searches"},
     *     security={{"bearerAuth":{}}},
     *     @OA\Parameter(
     *          name="firm_id",
     *          in="header",
     *          required=true,
     *          @OA\Schema(
     *              type="integer",
     *              example=1
     *          )
     *      ),
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="search_query",
     *                     type="string",
     *                     example="Параце"
     *                 ),
     *                 @OA\Property(
     *                     property="product_type_id",
     *                     type="integer",
     *                     example=1
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="Success",
     *         @OA\MediaType(
     *           mediaType="application/json",
     *      )
     *     ),
     *     @OA\Response(
     *         response=401,
     *         description="Unauthenticated",
     *         @OA\JsonContent(
     *             type="object",
     *             @OA\Property(property="message", type="string", example="Unauthenticated")
     *         )
     *     )
     * )
     */
    public function search(array $params): LengthAwarePaginator|array
    {
        $client = app('elasticsearch');
        $from = ($params['page'] - 1) * $params['per_page'];

        $search_query = mb_strtolower(trim($params['search_query']));
        if (mb_strlen($search_query) < 3) {
            return [];
        }

        $q_words = explode(' ', trim($search_query));
        $productType = $this->productTypeService->typeById($params['product_type_id']);
        $checkedCityIds = $this->cityInterface->getCheckedCityIds($params['firm_id']);
        $firmIds = $this->partnerInterface->getCheckedPartnerIds($params['firm_id'], $checkedCityIds);
        $overheadIds = Overhead::whereIn('firm_id', $firmIds)->pluck('id')->toArray();

        $mustQueries = [];
        $filters = [];

        $filters[] = [
            'terms' => [
                'overhead_id' => $overheadIds
            ],
        ];

        if ($params['discount']) {
            $filters[] = [
                'range' => [
                    'discounts' => [
                        'gt' => 0
                    ]
                ]
            ];
        }

        if (!empty($params['price_min']) || !empty($params['price_max'])) {
            $priceRange = [];
            if (!empty($params['price_min'])) {
                $priceRange['gte'] = (float)$params['price_min'];
            }
            if (!empty($params['price_max'])) {
                $priceRange['lte'] = (float)$params['price_max'];
            }
            $filters[] = [
                "range" => [
                    "price" => $priceRange
                ]
            ];
        }

        $sortBy = ['_score' => 'desc'];
        if ($params['sort']) {
            if ($params['sort'] == 'price_desc') {
                $sortBy = ['price' => 'desc'];
            } else if ($params['sort'] == 'price_asc') {
                $sortBy = ['price' => 'asc'];
            } else if ($params['sort'] == 'rating_desc') {
                $sortBy = ['rating' => 'desc'];
            }
        }

        $productTypeQueries = [];
        if (!empty($productType)) {
            foreach ($productType as $kw) {
                $cleanKw = trim($kw);
                if (!empty($cleanKw)) {
                    $productTypeQueries[] = [
                        'match' => [
                            'title' => [
                                'query' => $kw,
                            ]
                        ]
                    ];
                }
            }
        }
        if (!empty($productTypeQueries)) {
            $filters[] = [
                'bool' => [
                    'should' => $productTypeQueries,
                    'minimum_should_match' => 1
                ]
            ];
        }

        $shouldQueries = [];
        $boost_factor = 5.0;
        if (!empty($search_query)) {
            foreach ($q_words as $index => $word) {
                $fuzziness = null;
                if (preg_match('/[а-яё]/i', $word)) {
                    $current_boost = $boost_factor + 2;
                    $fuzziness = 'AUTO';
                } else {
                    $current_boost = $boost_factor * 0.5;
                    $fuzziness = 0;
                }

                $shouldQueries[] = [
                    'match_bool_prefix' => [
                        'title' => [
                            'query' => $word,
                            'fuzziness' => $fuzziness,
                            'boost' => $current_boost,
                        ]
                    ]
                ];
            }

        }
        $param = [
            'index' => 'drugstore',
            'body' => [
                'from' => $from,
                'size' => $params['per_page'],
                'query' => [
                    'bool' => [
                        'must' => $mustQueries,
                        'should' => $shouldQueries,
                        'minimum_should_match' => !empty($mustQueries) ? 0 : (!empty($shouldQueries) ? 1 : 0),
                        'filter' => $filters,
                    ],
                ],
                'sort' => $sortBy,
            ]
        ];

        $res = $client->search($param);
        $items = collect($res['hits']['hits'])->pluck('_source')->values();

        // fixme: Append real quantity here
        $items = $items->map(function ($item) {
            $item['quantity'] = 0;
            return $item;
        });
        $total = $res['hits']['total']['value'];

        return new \Illuminate\Pagination\LengthAwarePaginator(
            $items,
            $total,
            $params['per_page'],
            $params['page'],
            ['path' => request()->url(), 'query' => request()->query()]
        );

    }
}


